import torch as t
import ast
from transformers import AutoProcessor, AutoModelForImageTextToText, AutoTokenizer
import argparse
import pandas as pd
from tqdm import tqdm
from pathlib import Path
from datasets import load_dataset
from .utils import filter_dataset
from huggingface_hub import login, hf_hub_download
import os
import dotenv

dotenv.load_dotenv(override=True)

# load huggingface token from .env file and login
login(token=os.getenv("HF_TOKEN"))

print("token:", os.getenv("HF_TOKEN"))

def load_statements(dataset_name, model, contrast=None):
    """
    Load statements and labels from csv file, return list of strings and labels.
    """
    statements = []
    labels = []
    
    if "geometry_of_truth" in dataset_name:
        dataset = pd.read_csv(dataset_name)

        for _, row in dataset.iterrows():
            if contrast == True:
                pseudo_label = " This statement is true."
            elif contrast == False:
                pseudo_label = " This statement is false."
            elif contrast == None:
                pseudo_label = ""
                
            messages = [
                {"role": "system", "content": str(row['statement'])},
                {"role": "user", "content": str(row['statement']) + pseudo_label},
            ]
            labels = dataset['label'].tolist()
    else:
        path = hf_hub_download(
            repo_id="XYZ-XXXX-3/dishonesty-bench",
            filename=f"{dataset_name}.csv",
            repo_type="dataset",
            token=os.getenv("HF_TOKEN")
        )
        print("Loading dataset from:", path)
        
        dataset = filter_dataset(pd.read_csv(path), model)

        # check if dishonest column has all the same values
        if dataset['dishonest'].nunique() == 1:
            print(f"Info: {dataset_name} - all labels are the same")

        for _, row in dataset.iterrows():
            messages = ast.literal_eval(row['messages'])
            statements.append(messages.copy())
            labels.append(row['dishonest'])

    processed_statements = processor.apply_chat_template(statements, tokenize=False, add_generation_prompt=False)

    # strip off the trailing end-of-sentence token
    tokenizer = AutoTokenizer.from_pretrained("mistralai/Mistral-Small-3.1-24B-Instruct-2503")
    processed_statements = [s[:s.rindex(tokenizer.eos_token)] for s in processed_statements]

    return processed_statements, labels

def get_activations(statements, model, processor, layers, last_token=False):
    """
    Get given layer activations for multimodal models.
    Return dictionary of stacked activations.
    """
    activations = {}
    
    # Process inputs with the processor
    inputs = processor(text=statements, return_tensors="pt", padding=True, truncation=True)
    inputs = {k: v.to(model.device) for k, v in inputs.items()}
    
    # Extract activations
    with t.set_grad_enabled(False):
        outputs = model(**inputs, output_hidden_states=True)
        
        # Handle different output structures
        if hasattr(outputs, 'hidden_states'):
            hidden_states = outputs.hidden_states
        elif hasattr(outputs, 'encoder_hidden_states'):
            hidden_states = outputs.encoder_hidden_states
        else:
            hidden_states = outputs['hidden_states'] if 'hidden_states' in outputs else outputs['encoder_hidden_states']
            
        # Extract layer activations
        for layer in layers:
            if layer < len(hidden_states):
                layer_activations = hidden_states[layer]
            else:
                # Use last available layer if index is too high
                layer_activations = hidden_states[-1]
            
            if last_token:
                # Get only the last token's activations
                layer_activations = layer_activations[:, -1, :]
            
            activations[layer] = layer_activations

    return activations

def activations_exist(save_dir, batch_idx, layers):
    """
    Check if activations, labels, and token maps already exist for a given batch and layers.
    """
    if not save_dir.exists():
        return False
        
    # Check if all layer files and label file exist for this batch
    files_exist = all(
        (save_dir / f"layer_{layer}_{batch_idx}.pt").exists() 
        for layer in layers
    )
    labels_exist = (save_dir / f"labels_{batch_idx}.pt").exists()
    
    return files_exist and labels_exist

if __name__ == "__main__":
    """
    read statements from dataset, record activations in given layers, and save to specified files
    """
    parser = argparse.ArgumentParser(description="Generate activations for Mistral models")
    parser.add_argument("--model", default="mistralai/Mistral-Small-3.1-24B-Instruct-2503",
                        help="Mistral model to use")
    parser.add_argument("--layers", nargs='+', type=int,
                        help="Layers to save embeddings from")
    parser.add_argument("--datasets", nargs='+',
                        help="Names of datasets, without .csv extension")
    parser.add_argument("--output_dir", default="activations",
                        help="Directory to save activations to")
    parser.add_argument("--batch_size", type=int, default=1,
                        help="Number of examples to process at once (reduce if running into memory issues)")
    parser.add_argument("--contrast", action="store_true", help="Whether to use contrast statements", default=False)
    args = parser.parse_args()

    print("args:", args)

    # Load multimodal model and processor
    print(f"Loading multimodal model: {args.model}")
    model = AutoModelForImageTextToText.from_pretrained(
        args.model,
        torch_dtype=t.bfloat16,
        device_map="auto",
        trust_remote_code=True,
        token=os.getenv("HF_TOKEN")
    )

    processor = AutoProcessor.from_pretrained(args.model, trust_remote_code=True, token=os.getenv("HF_TOKEN"))
    
    print(f"Successfully loaded: {args.model}")
    print(f"Model type: {type(model).__name__}")
    print(f"Processor type: {type(processor).__name__}")
    
    contrasts = [False, True] if args.contrast else [None]
    for dataset in args.datasets:
        for contrast in contrasts:
            print(f"Extracting activations for {dataset} with contrast {contrast}")
            statements, labels = load_statements(dataset_name=dataset, contrast=contrast, model=args.model)
            labels = t.tensor(labels, dtype=t.bfloat16)

            layers = args.layers
            if not layers:
                # Try to determine the number of layers
                if hasattr(model.config, "num_hidden_layers"):
                    last_layer = model.config.num_hidden_layers - 1
                elif hasattr(model.config, "n_layers"):
                    last_layer = model.config.n_layers - 1
                else:
                    # Default for Mistral models
                    last_layer = 30
                    
                # 25th percentile layer
                layers = [last_layer // 4]

            print("Layers to extract:", layers)

            save_dir = Path(f"{args.output_dir}") / args.model
            if not save_dir.exists():
                save_dir.mkdir(parents=True)

            dataset_name = dataset.split("/")[-1].split(".csv")[0]
            save_dir = save_dir / dataset_name

            if not save_dir.exists():
                save_dir.mkdir(parents=True)

            if contrast:
                safe_path = save_dir / f"contrast_{contrast}"
            else:
                safe_path = save_dir / "No_Contrast"
                
            # Check if all activations already exist
            all_batches_exist = all(
                activations_exist(safe_path, idx, layers)
                for idx in range(0, len(statements), args.batch_size)
            )
            
            if all_batches_exist:
                print(f"Skipping {dataset} with contrast {contrast} - activations already exist")
                continue

            print("Extracting activations for layers", layers)
            batch_size = args.batch_size
            
            for idx in tqdm(range(0, len(statements), batch_size)):
                if activations_exist(safe_path, idx, layers):
                    continue
                    
                activations = get_activations(
                    statements[idx:idx + batch_size],
                    model,
                    processor,
                    layers
                )

                for layer, act in activations.items():
                    act = act.to(t.bfloat16)
                    safe_path.mkdir(parents=True, exist_ok=True)
                    t.save(act, safe_path / f"layer_{layer}_{idx}.pt")
                
                t.save(labels[idx:idx + batch_size], safe_path / f"labels_{idx}.pt")
                   
            print("Saved activations, labels", safe_path) 